package teamroots.embers.tileentity;

import net.minecraft.block.state.IBlockState;
import net.minecraft.client.resources.I18n;
import net.minecraft.entity.Entity;
import net.minecraft.entity.item.EntityItem;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.network.NetworkManager;
import net.minecraft.network.play.server.SPacketUpdateTileEntity;
import net.minecraft.util.EntitySelectors;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.EnumHand;
import net.minecraft.util.ITickable;
import net.minecraft.util.math.AxisAlignedBB;
import net.minecraft.util.math.BlockPos;
import net.minecraft.world.World;
import net.minecraftforge.common.capabilities.Capability;
import net.minecraftforge.fluids.Fluid;
import net.minecraftforge.fluids.FluidStack;
import net.minecraftforge.fluids.FluidTank;
import net.minecraftforge.fluids.FluidUtil;
import net.minecraftforge.fluids.capability.CapabilityFluidHandler;
import net.minecraftforge.items.CapabilityItemHandler;
import net.minecraftforge.items.ItemStackHandler;
import teamroots.embers.api.tile.IExtraCapabilityInformation;
import teamroots.embers.config.ConfigMachine;
import teamroots.embers.particle.ParticleUtil;
import teamroots.embers.util.FluidColorHelper;
import teamroots.embers.util.Misc;

import javax.annotation.Nullable;
import java.awt.*;
import java.util.List;
import java.util.Random;

public class TileEntityFurnaceTop extends TileEntityOpenTank implements ITileEntityBase, ITickable, IExtraCapabilityInformation {
    public static int capacity = ConfigMachine.MELTER.tankSize;
    public double angle = 0;
    int ticksExisted = 0;

    public ItemStackHandler inventory = new ItemStackHandler(1) {
        @Override
        protected void onContentsChanged(int slot) {
            // We need to tell the tile entity that something has changed so
            // that the chest contents is persisted
            TileEntityFurnaceTop.this.markDirty();
        }
    };

    public TileEntityFurnaceTop() {
        super();
        tank = new FluidTank(capacity) {
            @Override
            public void onContentsChanged() {
                TileEntityFurnaceTop.this.markDirty();
            }

            @Override
            public int fill(FluidStack resource, boolean doFill) {
                if (Misc.isGaseousFluid(resource)) {
                    setEscapedFluid(resource);
                    return resource.amount;
                }
                return super.fill(resource, doFill);
            }
        };
        tank.setTileEntity(this);
        tank.setCanFill(true);
        tank.setCanDrain(true);
    }

    @Override
    public NBTTagCompound writeToNBT(NBTTagCompound tag) {
        super.writeToNBT(tag);
        tag.setTag("inventory", inventory.serializeNBT());
        return tag;
    }

    @Override
    public void readFromNBT(NBTTagCompound tag) {
        super.readFromNBT(tag);
        if (tag.hasKey("inventory")) {
            inventory.deserializeNBT(tag.getCompoundTag("inventory"));
        }
    }

    @Override
    public NBTTagCompound getUpdateTag() {
        return writeToNBT(new NBTTagCompound());
    }

    @Nullable
    @Override
    public SPacketUpdateTileEntity getUpdatePacket() {
        return new SPacketUpdateTileEntity(getPos(), 0, getUpdateTag());
    }

    @Override
    public void onDataPacket(NetworkManager net, SPacketUpdateTileEntity pkt) {
        readFromNBT(pkt.getNbtCompound());
    }

    @Override
    public boolean activate(World world, BlockPos pos, IBlockState state, EntityPlayer player, EnumHand hand,
                            EnumFacing side, float hitX, float hitY, float hitZ) {
        ItemStack heldItem = player.getHeldItemMainhand();
        if (!heldItem.isEmpty()) {
            boolean didFill = FluidUtil.interactWithFluidHandler(player, hand, this.getCapability(CapabilityFluidHandler.FLUID_HANDLER_CAPABILITY, side));

            if (didFill) {
                this.markDirty();
                return true;
            } else {
                player.setHeldItem(hand, this.inventory.insertItem(0, heldItem, false));
                markDirty();
                return true;
            }
        } else {
            if (!inventory.getStackInSlot(0).isEmpty() && !world.isRemote) {
                world.spawnEntity(new EntityItem(world, player.posX, player.posY, player.posZ, inventory.getStackInSlot(0)));
                inventory.setStackInSlot(0, ItemStack.EMPTY);
                markDirty();
                return true;
            }
        }
        return false;
    }

    @Override
    public void markDirty() {
        super.markDirty();
        Misc.syncTE(this);
    }

    public int getCapacity() {
        return tank.getCapacity();
    }

    public FluidStack getFluidStack() {
        return tank.getFluid();
    }

    public FluidTank getTank() {
        return tank;
    }

    @Deprecated
    public Fluid getFluid() {
        if (tank.getFluid() != null) {
            return tank.getFluid().getFluid();
        }
        return null;
    }

    @Deprecated
    public int getAmount() {
        return tank.getFluidAmount();
    }


    @Override
    public void breakBlock(World world, BlockPos pos, IBlockState state, EntityPlayer player) {
        this.invalidate();
        Misc.spawnInventoryInWorld(world, pos.getX() + 0.5, pos.getY() + 0.5, pos.getZ() + 0.5, inventory);
        world.setTileEntity(pos, null);
    }

    @Override
    public boolean hasCapability(Capability<?> capability, EnumFacing facing) {
        if (capability == CapabilityItemHandler.ITEM_HANDLER_CAPABILITY) {
            return true;
        }
        return super.hasCapability(capability, facing);
    }

    @Override
    public <T> T getCapability(Capability<T> capability, EnumFacing facing) {
        if (capability == CapabilityItemHandler.ITEM_HANDLER_CAPABILITY) {
            return (T) this.inventory;
        }
        return super.getCapability(capability, facing);
    }

    @Override
    public void update() {
        angle++;
        ticksExisted++;
        if (ticksExisted % 10 == 0) {
            List<EntityItem> items = onGetEntitiesWithinAABB(getWorld(), EntityItem.class, new AxisAlignedBB(getPos().getX(), getPos().getY(), getPos().getZ(), getPos().getX() + 1, getPos().getY() + 1.25, getPos().getZ() + 1));
            for (EntityItem item : items) {
                ItemStack stack = inventory.insertItem(0, item.getItem(), false);
                if (!stack.isEmpty()) {
                    item.setItem(stack);
                } else {
                    getWorld().removeEntity(item);
                }
            }
        }
        if (world.isRemote && shouldEmitParticles())
            updateEscapeParticles();
    }

    @Override
    protected void updateEscapeParticles() {
        Color fluidColor = new Color(FluidColorHelper.getColor(lastEscaped), true);
        Random random = new Random();
        for (int i = 0; i < 3; i++) {
            float xOffset = 0.5f + (random.nextFloat() - 0.5f) * 2 * 0.2f;
            float yOffset = 0.9f;
            float zOffset = 0.5f + (random.nextFloat() - 0.5f) * 2 * 0.2f;

            ParticleUtil.spawnParticleVapor(world, pos.getX() + xOffset, pos.getY() + yOffset, pos.getZ() + zOffset, 0, 1 / 20f, 0, fluidColor.getRed() / 255f, fluidColor.getGreen() / 255f, fluidColor.getBlue() / 255f, fluidColor.getAlpha() / 255f, 4, 2, 20);
        }
    }

    @Override
    public boolean hasCapabilityDescription(Capability<?> capability) {
        return capability == CapabilityFluidHandler.FLUID_HANDLER_CAPABILITY || capability == CapabilityItemHandler.ITEM_HANDLER_CAPABILITY;
    }

    @Override
    public void addCapabilityDescription(List<String> strings, Capability<?> capability, EnumFacing facing) {
        if (capability == CapabilityItemHandler.ITEM_HANDLER_CAPABILITY)
            strings.add(IExtraCapabilityInformation.formatCapability(EnumIOType.INPUT, "embers.tooltip.goggles.item", null));
        if (capability == CapabilityFluidHandler.FLUID_HANDLER_CAPABILITY)
            strings.add(IExtraCapabilityInformation.formatCapability(EnumIOType.OUTPUT, "embers.tooltip.goggles.fluid", I18n.format("embers.tooltip.goggles.fluid.metal")));
    }

    private <T extends Entity> List<T> onGetEntitiesWithinAABB(World world, Class<T> entityClass, AxisAlignedBB axisAlignedBB) {
        return world.getEntitiesWithinAABB(entityClass, axisAlignedBB, entity -> EntitySelectors.NOT_SPECTATING.apply(entity) && entity.isEntityAlive());
    }
}
